/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "rf_system_4channel.h"
#include <QCoreApplication>
#include <QDebug>
#include <QMessageBox>


//
//TODO:
//it would probably be best if the constructor class were static as well.
//


#ifdef ETHERCAT
	RF_System_4channel::RF_System_4channel() : ethercatModule(*this)
#else
	RF_System_4channel::RF_System_4channel()
#endif
{
	//
	//TODO:
	//Convert to use channel_count from config instead of inflexible CHANNELCOUNT define.
	//

	qDebug() << "RF_System_4channel: [Reading config file]";
	config = new ConfigHandler(QCoreApplication::applicationDirPath() + "/config.txt");

	/* Set up the rotary switch */
	int pos = -1;

	QStringList switch_pins;
	QString switch_pins_string = config->get_rotary_switch_pins();
	if (!switch_pins_string.isEmpty())
	{
		/* Note: Only linux/RPi actually supports the rotary switch. */
		switch_pins = switch_pins_string.split(",");
		rotary_switch = new RotarySwitch(switch_pins);

		/* Retry reading the switch up to 20 times; break the look on the first valid value that's been read */
		for (int i = 0; i < 20; i++)
		{
			pos = rotary_switch->get_Position();

			if (pos >= 0)
			{
				break;
			}
		}

		/* If no valid switch value was read, show an error message and exit the GUI */
		if (pos < 0)
		{
			QMessageBox message;
			message.critical(nullptr, "Rotary Switch failure", "Failed to get valid rotary switch value");
			exit(0);
		}
	}
	else
	{
		pos = 0;
	}

	QStringList port_list = probe_serialports();

	/* Set up a single port for RCM */
	RCM_USB_port = new Serial();
	RCM_USB_port->setup(config->get_RCM_port_USB(), 0);
	connect(RCM_USB_port->serial, &QSerialPort::readyRead, this, &RF_System_4channel::RCM_USB_message_handler);
//	connect(RCM_USB_port->serial, &QSerialPort::errorOccurred, this, &RF_System_4channel::RCM_USB_error_handler);

	threadList = new QList<QThread*>();

	for (int i = 0; i < channel_count; i++)
	{
		channel = new RF_Channel(i+1, i+1 + pos*CHANNELCOUNT, port_list.at(i));
		RF_System::Channels->append(channel);

		connect(this, &RF_System_4channel::signal_RCM_USB_message, RF_System::Channels->at(i), &RF_Channel::RCM_Live_serial_writeRead_to_SGx);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_RCM_serial_response, this, &RF_System_4channel::RCM_USB_response_handler);

		channelThread = new QThread();
		threadList->append(channelThread);

		connect(channelThread, &QThread::finished, channel, &QObject::deleteLater);
		channel->moveToThread(channelThread);

		channelThread->start();
	}
}

RF_System_4channel::~RF_System_4channel()
{
	for (int i = 0; i < channel_count; i++)
	{
		threadList->at(i)->quit();
		threadList->at(i)->wait();
	}
}

int RF_System_4channel::channelCount()
{
	return channel_count;
}

/* Check available serial ports.
 * Correlate the correct SGx board to the correct channel. */
QStringList RF_System_4channel::probe_serialports()
{
	//The amount of PORTS provided in the config file may be less than the amount of channels to be initialized, in which case the Channels without a port remain dormant.
	QStringList comport_list = config->get_output_port().split(",");
	qDebug() << comport_list;

	int ports_count = comport_list.count();
	int channel_count = config->get_channel_count();

	QStringList ordered_comport_list;
	for (int i = 0; i < CHANNELCOUNT; i++)
	{
		ordered_comport_list.append("");
	}

	// Perform this loop for as many ports as are provided in the config, not the amount that the GUI expects; otherwise 'index out of range' crashes occur.
	for (int k = 0; k < ports_count; k++)
	{
		Serial serialport;
		serialport.setup(comport_list.at(k), 0);

		// If the port in question cannot be opened (for example because it's already in use), continue to the next iteration of the from loop.
		if (!serialport.open())
		{
			qDebug() << serialport.serial->portName() << "Could not be opened. Skipping";
			continue;
		}

		QString rx = "";
		serialport.serialWrite("$IDN,0\r\n");

		//
		// TODO:
		// This while loop is potentially an infinite
		//

		while(!rx.contains("\r\n") && !rx.contains("$IDN,"))
		{
			serialport.serial->waitForReadyRead(1000);
			rx += serialport.serialRead();
		}

		serialport.close();		//Done probing; close the serialport so that it can be used by other parts of the software.

		QStringList RX = rx.split(",");
		int temp_channel_num = RX.at(1).toInt();
		int switch_position_count;

		if (config->get_rotary_switch_pins() != "")
		{
			//Correlate switch position count to the amount of pins defined in config (assuming each pin is one possible position)
			switch_position_count = config->get_rotary_switch_pins().split(",").count();
		}
		else
		{
			//For debugging convenience when switch is unavailable
			switch_position_count = 4;
		}

		for (int i = 0; i < channel_count; i++)
		{
			/* Application can find a channel in one of two states:
			 * 1. The basic channel number (at power cycle)
			 * 2. The basic channel number OFFSET by any switch position * port_count (after restarting GUI / Changing switch value).
			 * Look for either channel number value, it's guaranteed to be unique if everything is wired up correctly */

			for (int j = 0; j < switch_position_count; j++)
			{
//				qDebug() << k << temp_channel_num << i+1 << (i+1) + j * CHANNELCOUNT;
				if (temp_channel_num == (i+1) || temp_channel_num == (i+1) + j * CHANNELCOUNT)
				{
					ordered_comport_list[i] = serialport.serial->portName();
				}
			}
		}
	}

	for (int i = 0; i < CHANNELCOUNT; i++)
	{
		qDebug() << "Subsystem[" << i << "] = " << ordered_comport_list.at(i);
	}

	return ordered_comport_list;
}

/**********************************************************************************************************************************************************************************
 * Remote Command Mode: Exchange messages between RCM port and SGX port.
 *********************************************************************************************************************************************************************************/

//readyRead handler
void RF_System_4channel::RCM_USB_message_handler()
{
	static QString received_message = "";

	received_message += RCM_USB_port->serialRead();
	if ((received_message.contains("\r\n") || received_message.contains("\r") || received_message.contains("\n")) && received_message.contains("$"))
	{
		qDebug() << "RCM USB: " << received_message;
	}
	else
	{
		return;
	}

	/* Make sure only one command is handled at a time */
	if (received_message.count("$") > 1)
	{
		RCM_USB_port->serialWrite("RCM: One command at a time please.");
		received_message = ""; //Message was sent; clear the variable
		return;
	}

	if (!received_message.contains(QRegExp("\\$\\w{1,}\r*\n*\r*")))
	{
		qDebug() << "RCM: Command not valid.";
		RCM_USB_port->serialWrite("RCM: Command not valid.");
		received_message = ""; //Message was sent; clear the variable
		return;
	}

	// Disallow commands with multi-line returns and other stuff that doesn't conform to the protocol properly.
	QStringList banlist
	{
		"SWP", "SWPD",
		"HELP", "HELPDEV",
		"CHANS", "CHANG",
		"IDLES",
		"EECSP", "EECSA"
	};

	for (int i = 0; i < banlist.count(); i++)
	{
		if (received_message.contains(("$"+banlist.at(i)), Qt::CaseInsensitive))
		{
			qDebug() << "RCM: This command is currently not supported by RCM";
			RCM_USB_port->serialWrite("RCM: This command is currently not supported by RCM");
			received_message = ""; //Message was sent; clear the variable
			return;
		}
	}

	emit signal_RCM_USB_message(received_message);
	received_message = ""; //Message was sent; clear the variable
}

void RF_System_4channel::RCM_USB_response_handler(QString response)
{
	//
	// TODO:
	// This is a workaround for the excessive newlines. Ideally RCM should have its own write functions (and probably its own class).
	//

#error Fixed the need for this 'workaround' in RCM_Class. However the RCM implementation in 4Channel is still using Serial instead of RCM_Class...
	response.replace("\r", "");
	response.replace("\n", "");
	RCM_USB_port->serialWrite(response);
}

void RF_System_4channel::RCM_USB_error_handler(QSerialPort::SerialPortError error)
{
//	if (error != QSerialPort::SerialPortError::NoError)
//	{
//		if (RCM_USB_port->serial->isOpen())
//		{
//			RCM_USB_port->close();
//		}

//		RCM_USB_port->open();
//	}
}
